<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use App\Models\UserBalance;
use Illuminate\Http\Request;

class TransitExpenseController extends Controller
{
    public function index()
    {
        $expenses = Transaction::where('type', 'transit_expense')
            ->where('user_id', auth()->id())
            ->latest()
            ->get();
        return view('transit-expenses.index', compact('expenses'));
    }

    public function create()
    {
        return view('transit-expenses.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'source' => 'required|in:cash,card',
            'description' => 'nullable|string',
        ]);

        $user = auth()->user();
        $balance = UserBalance::firstOrCreate(
            ['user_id' => $user->id],
            ['balance' => 0.00]
        );

        if ($balance->balance < $validated['amount']) {
            return redirect()->route('transit-expenses.index')->with('error', 'Balansingiz yetarli emas: ' . number_format($validated['amount'], 2) . ' so‘m kerak, sizda ' . number_format($balance->balance, 2) . ' so‘m bor.');
        }

        Transaction::create([
            'user_id' => $user->id,
            'type' => 'transit_expense',
            'amount' => $validated['amount'],
            'source' => $validated['source'],
            'description' => $validated['description'],
        ]);

        $balance->balance -= $validated['amount'];
        $balance->save();

        return redirect()->route('transit-expenses.index')->with('success', 'Xarajat qo‘shildi');
    }

    public function edit(Transaction $transaction)
    {
        if ($transaction->type !== 'transit_expense' || $transaction->user_id !== auth()->id()) {
            return redirect()->route('transit-expenses.index')->with('error', 'Bu tranzaksiya xarajat emas yoki sizga tegishli emas');
        }
        return view('transit-expenses.edit', compact('transaction'));
    }

    public function update(Request $request, Transaction $transaction)
    {
        if ($transaction->type !== 'transit_expense' || $transaction->user_id !== auth()->id()) {
            return redirect()->route('transit-expenses.index')->with('error', 'Bu tranzaksiya xarajat emas yoki sizga tegishli emas');
        }

        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'source' => 'required|in:cash,card',
            'description' => 'nullable|string',
        ]);

        $user = auth()->user();
        $balance = UserBalance::where('user_id', $user->id)->first();

        if (!$balance) {
            return redirect()->route('transit-expenses.index')->with('error', 'Balans topilmadi.');
        }

        $oldAmount = $transaction->amount;
        $balanceDifference = $validated['amount'] - $oldAmount;

        if ($balance->balance < $balanceDifference) {
            return redirect()->route('transit-expenses.index')->with('error', 'Balansingiz yetarli emas: ' . number_format($balanceDifference, 2) . ' so‘m kerak, sizda ' . number_format($balance->balance, 2) . ' so‘m bor.');
        }

        $transaction->update([
            'amount' => $validated['amount'],
            'source' => $validated['source'],
            'description' => $validated['description'],
        ]);

        $balance->balance -= $balanceDifference;
        $balance->save();

        return redirect()->route('transit-expenses.index')->with('success', 'Xarajat yangilandi');
    }

    public function destroy(Transaction $transaction)
    {
        if ($transaction->type !== 'transit_expense' || $transaction->user_id !== auth()->id()) {
            return redirect()->route('transit-expenses.index')->with('error', 'Bu tranzaksiya xarajat emas yoki sizga tegishli emas');
        }

        $user = auth()->user();
        $balance = UserBalance::where('user_id', $user->id)->first();

        if ($balance) {
            $balance->balance += $transaction->amount;
            $balance->save();
        }

        $transaction->delete();
        return redirect()->route('transit-expenses.index')->with('success', 'Xarajat o‘chirildi');
    }
}