<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Supplier;
use Illuminate\Http\Request;

class ProductPaymentController extends Controller
{
    public function index()
    {
        $payments = Payment::where('type', 'product_payment')->with('supplier')->latest()->get();
        return view('product-payments.index', compact('payments'));
    }

    public function create()
    {
        $suppliers = Supplier::all();
        return view('product-payments.create', compact('suppliers'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'amount' => 'required|numeric|min:0',
            'source' => 'required|in:cash,card',
            'description' => 'nullable|string',
        ]);

        $payment = Payment::create([
            'supplier_id' => $request->supplier_id,
            'type' => 'product_payment',
            'amount' => $request->amount,
            'source' => $request->source,
            'description' => $request->description,
        ]);

        // Update supplier debt balance
        $supplier = $payment->supplier;
        $toSupplierTotal = Payment::where('supplier_id', $supplier->id)
                                 ->where('type', 'to_supplier')
                                 ->sum('amount');
        $productPaymentTotal = Payment::where('supplier_id', $supplier->id)
                                     ->where('type', 'product_payment')
                                     ->sum('amount');
        $supplier->updateDebt($toSupplierTotal, $productPaymentTotal);

        return redirect()->route('product-payments.index')
                        ->with('success', 'Mahsulot to‘lovi qo‘shildi va qarz balansi yangilandi');
    }

    public function edit(Payment $payment)
    {
        if ($payment->type !== 'product_payment') {
            return redirect()->route('product-payments.index')->with('error', 'Bu to‘lov mahsulot to‘lovi emas');
        }
        $suppliers = Supplier::all();
        return view('product-payments.edit', compact('payment', 'suppliers'));
    }

    public function update(Request $request, Payment $payment)
    {
        if ($payment->type !== 'product_payment') {
            return redirect()->route('product-payments.index')->with('error', 'Bu to‘lov mahsulot to‘lovi emas');
        }

        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'amount' => 'required|numeric|min:0',
            'source' => 'required|in:cash,card',
            'description' => 'nullable|string',
        ]);

        $payment->update([
            'supplier_id' => $request->supplier_id,
            'amount' => $request->amount,
            'source' => $request->source,
            'description' => $request->description,
        ]);

        // Update supplier debt balance
        $supplier = $payment->supplier;
        $toSupplierTotal = Payment::where('supplier_id', $supplier->id)
                                 ->where('type', 'to_supplier')
                                 ->sum('amount');
        $productPaymentTotal = Payment::where('supplier_id', $supplier->id)
                                     ->where('type', 'product_payment')
                                     ->sum('amount');
        $supplier->updateDebt($toSupplierTotal, $productPaymentTotal);

        return redirect()->route('product-payments.index')
                        ->with('success', 'To‘lov yangilandi va qarz balansi yangilandi');
    }

    public function destroy(Payment $payment)
    {
        if ($payment->type !== 'product_payment') {
            return redirect()->route('product-payments.index')->with('error', 'Bu to‘lov mahsulot to‘lovi emas');
        }

        $supplier = $payment->supplier;
        $payment->delete();

        // Update supplier debt balance
        $toSupplierTotal = Payment::where('supplier_id', $supplier->id)
                                 ->where('type', 'to_supplier')
                                 ->sum('amount');
        $productPaymentTotal = Payment::where('supplier_id', $supplier->id)
                                     ->where('type', 'product_payment')
                                     ->sum('amount');
        $supplier->updateDebt($toSupplierTotal, $productPaymentTotal);

        return redirect()->route('product-payments.index')
                        ->with('success', 'To‘lov o‘chirildi va qarz balansi yangilandi');
    }
}