<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use Illuminate\Http\Request;

class IncomeController extends Controller
{
    public function index()
    {
        $incomes = Transaction::where('type', 'income')->latest()->get();
        return view('income.index', compact('incomes'));
    }

    public function create()
    {
        $categories = ['Kassadan', 'Tashqi mablag\'', 'Savdo', 'Boshqa'];
        return view('income.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0',
            'source' => 'required|in:cash,card',
            'category' => 'required|in:Kassadan,Tashqi mablag\',Savdo,Boshqa',
            'description' => 'nullable|string',
        ]);

        Transaction::create([
            'type' => 'income',
            'category' => $request->category,
            'amount' => $request->amount,
            'source' => $request->source,
            'description' => $request->description,
        ]);

        return redirect()->route('income.index')->with('success', 'Kirim qo‘shildi');
    }

    public function edit(Transaction $transaction)
    {
        if ($transaction->type !== 'income') {
            return redirect()->route('income.index')->with('error', 'Bu tranzaksiya kirim emas');
        }
        $categories = ['Kassadan', 'Tashqi mablag\'', 'Savdo', 'Boshqa'];
        return view('income.edit', compact('transaction', 'categories'));
    }

    public function update(Request $request, Transaction $transaction)
    {
        if ($transaction->type !== 'income') {
            return redirect()->route('income.index')->with('error', 'Bu tranzaksiya kirim emas');
        }

        $request->validate([
            'amount' => 'required|numeric|min:0',
            'source' => 'required|in:cash,card',
            'category' => 'required|in:Kassadan,Tashqi mablag\',Savdo,Boshqa',
            'description' => 'nullable|string',
        ]);

        $transaction->update([
            'category' => $request->category,
            'amount' => $request->amount,
            'source' => $request->source,
            'description' => $request->description,
        ]);

        return redirect()->route('income.index')->with('success', 'Kirim yangilandi');
    }

    public function destroy(Transaction $transaction)
    {
        if ($transaction->type !== 'income') {
            return redirect()->route('income.index')->with('error', 'Bu tranzaksiya kirim emas');
        }
        $transaction->delete();
        return redirect()->route('income.index')->with('success', 'Kirim o‘chirildi');
    }
}