<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use App\Models\Payment;
use App\Models\Supplier;
use App\Models\Category;
use App\Models\UserBalance;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class DashboardController extends Controller
{
    protected function calculateBalance()
    {
        $userId = auth()->id();
        $totalIncome = Transaction::where('type', 'income')->where('user_id', $userId)->sum('amount');
        $totalExpense = Transaction::where('type', 'transit_expense')->where('user_id', $userId)->sum('amount');
        $totalSupplierPayment = Payment::where('type', 'payment_to_supplier')->where('user_id', $userId)->sum('amount');
        return $totalIncome - $totalExpense - $totalSupplierPayment;
    }

    public function index(Request $request)
    {
        $userId = auth()->id();
        $mainBalance = $this->calculateBalance();
        $suppliers = Supplier::all();
        $incomeCategories = Category::where('type', 'income')->get();
        $expenseCategories = Category::where('type', 'expense')->get();
        $query = $request->input('search');

        $transactions = Transaction::select(
            'transactions.id as id',
            'transactions.type',
            \DB::raw('CAST(transactions.amount AS DECIMAL(15,2)) as amount'),
            'transactions.source',
            'categories.name as related_entity',
            'transactions.description',
            'transactions.images',
            'transactions.created_at',
            \DB::raw("'transaction' as source_table"),
            \DB::raw('NULL as supplier_name'),
            'users.name as user_name'
        )
        ->leftJoin('categories', 'transactions.category_id', '=', 'categories.id')
        ->join('users', 'transactions.user_id', '=', 'users.id')
        ->where('transactions.user_id', $userId);

        $payments = Payment::select(
            'payments.id as id',
            'payments.type',
            \DB::raw('CAST(payments.amount AS DECIMAL(15,2)) as amount'),
            'payments.source',
            'suppliers.name as related_entity',
            'payments.description',
            'payments.images',
            'payments.created_at',
            \DB::raw("'payment' as source_table"),
            'suppliers.name as supplier_name',
            'users.name as user_name'
        )
        ->join('suppliers', 'payments.supplier_id', '=', 'suppliers.id')
        ->join('users', 'payments.user_id', '=', 'users.id')
        ->where('payments.user_id', $userId);

        $results = $transactions->union($payments);

        if ($query) {
            $results = $results->where(function ($q) use ($query) {
                $q->where('description', 'like', "%{$query}%")
                  ->orWhere('source', 'like', "%{$query}%")
                  ->orWhere('users.name', 'like', "%{$query}%");
            });
        }

        $results = $results->orderBy('created_at', 'desc')->get();

        return view('dashboard', compact('mainBalance', 'suppliers', 'incomeCategories', 'expenseCategories', 'results'));
    }

    public function storeIncome(Request $request)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'category_id' => 'required|exists:categories,id',
            'source' => 'required|string|in:cash,card',
            'description' => 'nullable|string|max:255',
            'images.*' => 'nullable|image|mimes:jpeg,png|max:5120',
        ]);

        $user = auth()->user();
        $images = [];
        if ($request->hasFile('images')) {
            if (count($request->file('images')) > 12) {
                return redirect()->route('dashboard')->with('error', 'Maksimal 12 ta rasm yuklash mumkin!');
            }
            foreach ($request->file('images') as $image) {
                $path = $image->store('images', 'public');
                $images[] = $path;
            }
        }

        $transaction = Transaction::create([
            'user_id' => $user->id,
            'type' => 'income',
            'amount' => $validated['amount'],
            'category_id' => $validated['category_id'],
            'source' => $validated['source'],
            'description' => $validated['description'],
            'images' => json_encode($images),
        ]);

        $balance = UserBalance::firstOrCreate(
            ['user_id' => $user->id],
            ['balance' => 0.00]
        );
        $balance->balance += $validated['amount'];
        $balance->save();

        return redirect()->route('dashboard')->with('success', 'Kirim muvaffaqiyatli qo‘shildi!');
    }

    public function storeTransitExpense(Request $request)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'category_id' => 'required|exists:categories,id',
            'source' => 'required|string|in:cash,card',
            'description' => 'nullable|string|max:255',
            'images.*' => 'nullable|image|mimes:jpeg,png|max:5120',
        ]);

        $user = auth()->user();
        $balance = UserBalance::firstOrCreate(
            ['user_id' => $user->id],
            ['balance' => 0.00]
        );

        if ($balance->balance < $validated['amount']) {
            return redirect()->route('dashboard')->with('error', 'Balansingiz yetarli emas: ' . number_format($validated['amount'], 2) . ' so‘m kerak, sizda ' . number_format($balance->balance, 2) . ' so‘m bor.');
        }

        $images = [];
        if ($request->hasFile('images')) {
            if (count($request->file('images')) > 12) {
                return redirect()->route('dashboard')->with('error', 'Maksimal 12 ta rasm yuklash mumkin!');
            }
            foreach ($request->file('images') as $image) {
                $path = $image->store('images', 'public');
                $images[] = $path;
            }
        }

        Transaction::create([
            'user_id' => $user->id,
            'type' => 'transit_expense',
            'amount' => $validated['amount'],
            'category_id' => $validated['category_id'],
            'source' => $validated['source'],
            'description' => $validated['description'],
            'images' => json_encode($images),
        ]);

        $balance->balance -= $validated['amount'];
        $balance->save();

        return redirect()->route('dashboard')->with('success', 'Tranzit xarajat muvaffaqiyatli qo‘shildi!');
    }

    public function storePaymentToSupplier(Request $request)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'supplier_id' => 'required|exists:suppliers,id',
            'source' => 'required|string|in:cash,card',
            'description' => 'nullable|string|max:255',
            'images.*' => 'nullable|image|mimes:jpeg,png|max:5120',
        ]);

        $user = auth()->user();
        $balance = UserBalance::firstOrCreate(
            ['user_id' => $user->id],
            ['balance' => 0.00]
        );

        if ($balance->balance < $validated['amount']) {
            return redirect()->route('dashboard')->with('error', 'Balansingiz yetarli emas: ' . number_format($validated['amount'], 2) . ' so‘m kerak, sizda ' . number_format($balance->balance, 2) . ' so‘m bor.');
        }

        $images = [];
        if ($request->hasFile('images')) {
            if (count($request->file('images')) > 12) {
                return redirect()->route('dashboard')->with('error', 'Maksimal 12 ta rasm yuklash mumkin!');
            }
            foreach ($request->file('images') as $image) {
                $path = $image->store('images', 'public');
                $images[] = $path;
            }
        }

        $payment = Payment::create([
            'user_id' => $user->id,
            'type' => 'payment_to_supplier',
            'amount' => $validated['amount'],
            'supplier_id' => $validated['supplier_id'],
            'source' => $validated['source'],
            'description' => $validated['description'],
            'images' => json_encode($images),
        ]);

        $supplier = Supplier::find($validated['supplier_id']);
        $supplier->debt_balance += $validated['amount'];
        $supplier->save();

        $balance->balance -= $validated['amount'];
        $balance->save();

        return redirect()->route('dashboard')->with('success', 'To‘lov muvaffaqiyatli qo‘shildi!');
    }

    public function storeProductPayment(Request $request)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'supplier_id' => 'required|exists:suppliers,id',
            'description' => 'nullable|string|max:255',
            'images.*' => 'nullable|image|mimes:jpeg,png|max:5120',
        ]);

        $user = auth()->user();
        $images = [];
        if ($request->hasFile('images')) {
            if (count($request->file('images')) > 12) {
                return redirect()->route('dashboard')->with('error', 'Maksimal 12 ta rasm yuklash mumkin!');
            }
            foreach ($request->file('images') as $image) {
                $path = $image->store('images', 'public');
                $images[] = $path;
            }
        }

        $payment = Payment::create([
            'user_id' => $user->id,
            'type' => 'product_payment',
            'amount' => $validated['amount'],
            'supplier_id' => $validated['supplier_id'],
            'description' => $validated['description'],
            'images' => json_encode($images),
        ]);

        $supplier = Supplier::find($validated['supplier_id']);
        $supplier->debt_balance -= $validated['amount'];
        $supplier->save();

        return redirect()->route('dashboard')->with('success', 'Mahsulot to‘lovi muvaffaqiyatli qo‘shildi!');
    }
}